Imports Microsoft.VisualBasic
Imports System.Math
Imports System

Module MapPointUtilities
#Region "DmsFormat"
   ' Utility function - loop up string in collection,
   ' return default value if not available
   Private Function strLookUp(ByVal colCollection As Collection, ByVal strKey As String, _
   ByVal strNA As String) As String
      strLookUp = strNA
      On Error Resume Next
      strLookUp = colCollection(strKey)
   End Function

   ' Utility function - loop up double in collection,
   ' return default value if not available
   Private Function dblLookUp(ByVal colCollection As Collection, ByVal strKey As String, _
   ByVal dblNA As Double) As Double
      dblLookUp = dblNA
      On Error Resume Next
      dblLookUp = colCollection(strKey)
   End Function

   ' Utility function - Initialize collections
   ' used for field data and format storage
   Private Sub SetupCollections(ByVal dblVal As Double, _
   ByVal colValues As Collection, _
   ByVal colFormats As Collection, _
   ByVal blnIsLatitude As Boolean)
      ' Compute various fields, fill
      ' collections with potential replacements
      Dim intTmp As Integer
      intTmp = 1

      ' Compute compass point index, get rid
      ' of dblVal sign
      If dblVal < 0 Then
         intTmp = IIf(blnIsLatitude, 2, 0)
         dblVal = -dblVal
      Else
         intTmp = IIf(blnIsLatitude, 3, 1)
      End If

      ' add compass point
      colValues.Add(intTmp, "Cp")
      colFormats.Add("W|E|S|N", "Cp")

      ' Compute and add integer degrees
      intTmp = Int(dblVal)
      colValues.Add(intTmp, "Di")
      colFormats.Add("##0", "Di")

      ' Compute and add decimal degrees
      colValues.Add(dblVal, "Dd")
      colFormats.Add("##0.0####", "Dd")

      ' Compute and add integer minutes
      Dim dblTmp As Double
      dblTmp = (dblVal - intTmp) * 60
      intTmp = Int(dblTmp)
      colValues.Add(intTmp, "Mi")
      colFormats.Add("#0", "Mi")

      ' Compute and add decimal minutes
      colValues.Add(dblTmp, "Md")
      colFormats.Add("#0.0##", "Md")

      ' Compute and add integer seconds
      dblTmp = (dblTmp - intTmp) * 60
      intTmp = Int(dblTmp)
      colValues.Add(intTmp, "Si")
      colFormats.Add("#0", "Si")

      ' Compute and add decimal seconds
      colValues.Add(dblTmp, "Sd")
      colFormats.Add("#0.0", "Sd")
   End Sub

   ' The actual working horse
   Public Function DmsFormat(ByVal dblVal As Double, ByVal strPattern As String, _
   ByVal blnIsLatitude As Boolean, _
   ByVal blnForceDp As Boolean) As String
      Dim colValues As New Collection()
      Dim colFormats As New Collection()

      ' Compute fields, init collections
      SetupCollections(dblVal, colValues, colFormats, blnIsLatitude)

      Dim strRes As String ' resulting string
      Dim nCurPos As Integer ' scan position
      Dim nLen As Integer ' pattern string length

      strRes = ""
      nLen = Len(strPattern)
      nCurPos = 1

      ' scan pattern, building result string
      Do While nCurPos <= nLen
         Dim nPosLt As Integer ' current less-than
         Dim nPosGt As Integer ' current greater-than

         ' Find next Less-than and Greater-than
         nPosLt = InStr(nCurPos, strPattern, "<")
         nPosGt = IIf(nPosLt = 0, 0, InStr(nPosLt + 1, strPattern, ">"))

         If nPosLt = 0 Or nPosGt = 0 Then
            ' no more '<' found, or '<' without '>' -
            ' in these cases, we're done
            ' add remainder to result
            strRes = strRes & Mid$(strPattern, nCurPos)

            ' set current position to string end
            nCurPos = nLen + 1
            Exit Do ' exit the loop
         End If

         ' Add anything preceding the '<' as-is to result
         strRes = strRes & Mid$(strPattern, nCurPos, nPosLt - nCurPos)

         If nPosGt - nPosLt < 3 Then
            ' Invalid sequence, add as-is, continue
            strRes = strRes & Mid$(strPattern, nPosLt, nPosGt - nPosLt + 1)
            nCurPos = nPosGt + 1
            ' VB doesn't have a continue statement
            ' emulate with GoTo
            GoTo Continue
         End If

         ' Retrieve the field or formatting command
         Dim strField As String
         strField = Mid$(strPattern, nPosLt + 1, 2)

         Dim strFormat As String

         ' Is there an assignment ?
         If Mid$(strPattern, nPosLt + 3, 1) = "=" Then
            ' yes - retrieve format string
            strFormat = Mid$(strPattern, nPosLt + 4, nPosGt - nPosLt - 4)
         Else
            ' no - get its initial value otherwise
            strFormat = strLookUp(colFormats, strField, "")
         End If

         Dim strFmtParts() As String
         Dim nWidth As Integer

         ' assume no width
         nWidth = 0

         ' split into actual format and width
         strFmtParts = Split(strFormat, ":")
         If UBound(strFmtParts) > 0 Then
            ' we have a width, reassign format
            strFormat = strFmtParts(0)
            nWidth = Val(strFmtParts(1))
         End If

         Dim strFormattedField
         strFormattedField = ""

         Select Case UCase$(strField)
            ' Handle special cases
         Case "CP" ' compass point
               Dim strCp() As String
               strCp = Split(strFormat, "|")
               If UBound(strCp) = 3 Then
                  strFormattedField = strCp(dblLookUp(colValues, "Cp", 0))
               End If
            Case "LT" ' Literal '<'
               strFormattedField = "<"
            Case "GT" ' Literal '>'
               strFormattedField = ">"
            Case Else ' Collection items
               Dim dblTheValue

               ' Get value corresponding to field
               dblTheValue = dblLookUp(colValues, strField, 99.99)
               If strFormat = "" Then
                  strFormat = strLookUp(colFormats, strField, "")
               End If
               strFormattedField = Format(dblTheValue, strFormat)
               If blnForceDp Then
                  ' replace comma with decimal field if requested
                  strFormattedField = Replace(strFormattedField, _
                                              ",", ".")
               End If
         End Select

         ' If there a non-zero width, right-pad or left-pad
         ' with spaces according to sign
         If nWidth <> 0 Then
            strFormattedField = IIf(nWidth > 0, _
               Right$(Space$(Abs(nWidth)) & strFormattedField, Abs(nWidth)), _
               Left$(strFormattedField & Space$(Abs(nWidth)), Abs(nWidth)))
            ' If you're wondering why Abs() is being used in the two cases,
            ' Vb seems to evaluate the two expressions first, then selects
            ' which one to use.
         End If

         ' add formatted field to result string
         strRes = strRes & strFormattedField

         ' move pointer beyond ">"
         nCurPos = nPosGt + 1

Continue:
      Loop

      DmsFormat = strRes
   End Function
#End Region

#Region "CalcPos"
   Sub CalcPos(ByVal objMap As MapPoint.Map, ByVal locX As MapPoint.Location, ByRef dblLat As Double, ByRef dblLon As Double)
      Static locNorthPole As MapPoint.Location
      Static locSantaCruz As MapPoint.Location ' Center of western hemisphere
      Static dblHalfEarth As Double ' Half circumference of the earth (as a sphere)
      Static dblQuarterEarth As Double ' Quarter circumference of the earth (as a sphere)
      ' Static Pi As Double

      'Check if initialization already done
      If locNorthPole Is Nothing Then
         locNorthPole = objMap.GetLocation(90, 0)
         locSantaCruz = objMap.GetLocation(0, -90)

         'Compute distance between north and south poles == half earth circumference
         dblHalfEarth = objMap.Distance(locNorthPole, objMap.GetLocation(-90, 0))

         'Quarter of that is the max distance a point may be away from locSantaCruz and still be in western hemisphere
         dblQuarterEarth = dblHalfEarth / 2
         ' Pi = 3.14159265358979
      End If

      'Compute latitude from distance to north pole
      dblLat = 90 - 180 * objMap.Distance(locNorthPole, locX) / dblHalfEarth

      Dim l As Double
      Dim d As Double

      'Compute great circle distance to locX from point on Greenwich meridian and computed Latitude
      d = objMap.Distance(objMap.GetLocation(dblLat, 0), locX)

      'convert latitude to radian
      l = (dblLat / 180) * Math.PI

      'Compute Longitude from great circle distance
      dblLon = 180 * Acos((Cos((d * 2 * Math.PI) / (2 * dblHalfEarth)) - Sin(l) * Sin(l)) / (Cos(l) * Cos(l))) / PI

      'Correct longitude sign if located in western hemisphere
      If objMap.Distance(locSantaCruz, locX) < dblQuarterEarth Then dblLon = -dblLon
   End Sub
#End Region
End Module
